#!/bin/sh

##################################################################
# Resolve CBS based path

if [ -z "$CBS_HOME" ]; then
EXE_DIR=`pwd`
SCRIPT_HOME=`dirname "$0"`
cd "$SCRIPT_HOME"
APP_BIN=`pwd`
APP_HOME=`dirname "$APP_BIN"`

CBS_HOME=$APP_HOME
echo "$CBS_HOME"
export APP_BIN APP_HOME CBS_HOME

cd "$CBS_HOME"
fi

# Define Temporary JAVA_HOME
# JAVA_HOME="../jre-linux-amd64"


##################################################################
# System Debug Flag, enabled by default (0 - Disable, 1 - Enable)
SYSTEM_DEBUG=1

##################################################################

# Use same size for Java Min and Max Heap size (0 - Disable, 1 - Enable)
JAVA_HEAP_MIN_MAX_SAME=0

# User defined java heap max in unit of MB
# Should be 1GB or less for 32 bit systems
JAVA_HEAP_MAX_IN_MB=1024

# Use maximum of 2GB for 64-bit system
# Can be set as large as physical memory, but we have set an upper limit to avoid memory dictation by the java process
JAVA64_HEAP_MAX_IN_MB=2048

##################################################################
# Since OBSR 5.5.5.0, we have integrated the shell scripts for the following OS-ARCH.
# OS : from `uname`
# ARCH: from `uname -m`

# Linux-i386 | Linux-i486 | Linux-i586 | Linux-i686 | Linux-amd64 | Linux-x86_64 | Linux-arm | Linux-ppc
# FreeBSD-i386 | FreeBSD-amd64
# OpenBSD-i386 | OpenBSD-amd64

SYSTEM_OS=`uname`
SYSTEM_ARCH=`uname -m`

SYSTEM_IDENTITY=`uname -i`
case "${SYSTEM_OS}" in
  OpenBSD* )
    # OpenBSD does not support "uname -i"
    ;;
  * )
    SYSTEM_IDENTITY=`uname -i`
    ;;
esac

SYSTEM_TYPE=
case "${SYSTEM_OS}" in
  Linux* )
    SYSTEM_TYPE="linux"
    ;;
  *BSD* )
    SYSTEM_TYPE="bsd"
    ;;
  Darwin* )
    SYSTEM_TYPE="macos"
    ;;
  SunOS )
    SYSTEM_TYPE="solaris"
    ;;
  * )
    SYSTEM_TYPE="unknown"
    ;;
esac

##################################################################
# "32 bit OS /Oracle SPARC" does not support in Ahsay CBS v7 --- task #10888
if [ "$SYSTEM_TYPE" = "solaris" ];
	then
		IS_32BIT_KERNEL="`isainfo -v | grep -c 64-bit`"
		if [ "$SYSTEM_TYPE" = "solaris" ] && [ "$IS_32BIT_KERNEL" = "0" ];
			then
			echo "'32 bit OS /Oracle SPARC' does not support in Ahsay CBS v7, Please refer to Ahsay support platform list for details."
			echo "Supported platforms: 'https://forum.ahsay.com/viewtopic.php?f=22&t=9795'"
			exit 0
		fi
fi

if [ "$SYSTEM_TYPE" = "solaris" ];
	then
		IS_SPARC_ARCH="`isainfo -v |grep -c sparc*`"
		if [ "$SYSTEM_TYPE" = "solaris" ] && [ "$IS_SPARC_ARCH" != "0" ];
			then
			echo "'32 bit OS /Oracle SPARC' does not support in Ahsay CBS v7, Please refer to Ahsay support platform list for details."
			echo "Supported platforms: 'https://forum.ahsay.com/viewtopic.php?f=22&t=9795'"
			exit 0
		fi
fi

##################################################################
# Unit Functions for all NIX OS

log_message() {
  echo "$1"
}

log_string() {
  case "${SYSTEM_TYPE}" in
    solaris )
      echo "$1\c"
      ;;
    * )
      echo -n "$1"
      ;;
  esac  
}

log_debug_message() {
  [ $SYSTEM_DEBUG -eq 1 ] && log_message "$1"
}

create_symlink_for_java_home() {
  if [ ! -n "$JAVA_HOME" ]; then
    log_message "Please set JAVA_HOME!"
    exit 1
  else
    log_message "Create JAVA_HOME symbolic link at $CBS_HOME/java"
    cd "$CBS_HOME"
    ln -sf "${JAVA_HOME}" "java"
    return 0
  fi
}

process_count_with_pid() {
  PID_VALUE=$1
  PS_OPTIONS=" "
  case "${SYSTEM_TYPE}" in
    solaris )
      PS_OPTIONS="-e"
      ;;
    * )
      ;;
  esac
  COUNT=`ps $PS_OPTIONS | awk -F" " '{print " "$1" "$2" "}' | grep -c " $PID_VALUE "`
  echo $COUNT
}

##################################################################
# Memory related functions

get_column_value() {
  local STRING_PATTERN="$1"
  local COL_NUM=$2
  echo $STRING_PATTERN | awk -F " " '{print $'"${COL_NUM}"'}'
}

get_physical_memory_in_kb() {
  case "${SYSTEM_TYPE}" in
    linux )
      linux_get_ram_info_in_kb
      ;;
    bsd )
      bsd_macos_get_ram_info_in_kb
      ;;
    macos )
      bsd_macos_get_ram_info_in_kb
      ;;
    solaris )
      solaris_get_ram_info_in_kb
      ;;
    * )
      echo "0"
      ;;
  esac
}

linux_get_ram_info_in_kb() {
  # Check the value by "cat /proc/meminfo"
  MEMORY_INFO=`cat /proc/meminfo | grep MemTotal:`
  TOTAL_MEMORY_IN_KB=`get_column_value "$MEMORY_INFO" 2`
  echo "$TOTAL_MEMORY_IN_KB"
}

bsd_macos_get_ram_info_in_kb() {
  REAL_MEM="`/sbin/sysctl -n hw.physmem`"
  REAL_MEM_IN_KB="`expr $REAL_MEM / 1024`"
  echo "${REAL_MEM_IN_KB}"
}

solaris_get_ram_info_in_kb() {
  REAL_MEM="`/usr/bin/kstat -p unix:0:system_pages:physmem | cut -f2`"
  PAGESIZE="`/usr/bin/pagesize`"
  REAL_MEM_IN_KB="`expr "$REAL_MEM" "*" "$PAGESIZE" "/" "1024"`"
  echo "${REAL_MEM_IN_KB}"
}

##################################################################
# Setup JAVA environment, including JAVA_HOME and the options limit

# Prepare JAVA_HOME
if [ -z "$JRE_HOME" ]; then
# If $CBS_HOME/java exist, it will always be used as JRE_HOME.
# Otherwise, we have to create symbolic link for the $CBS_HOME/java to the predefined JRE_HOME

if [ -d "$CBS_HOME/java" ];then
  JRE_HOME="$CBS_HOME/java"
else
  case "${SYSTEM_TYPE}" in
    # Modify JRE_HOME for Mac OS X
    # It is required to be here for this file will be used in upgrade patch as well
    macos)
      JRE_HOME=/System/Library/Frameworks/JavaVM.framework/Home
      ln -sf "$JRE_HOME" "$CBS_HOME/java"
      ;;
    linux)
      #  use bundled java if available
      case "${SYSTEM_ARCH}" in
        i[3-6]86)
          if [ -d $CBS_HOME/java-linux-x86 ]; then
            JRE_HOME=$CBS_HOME/java-linux-x86
            cd $CBS_HOME
            ln -sf "java-linux-x86" "java"
          else
            create_symlink_for_java_home
          fi
          ;;
        x86_64 | amd64)
          if [ -d $CBS_HOME/java-linux-x64 ]; then
            JRE_HOME=$CBS_HOME/java-linux-x64
            ln -sf "java-linux-x64" "java"
          else
            create_symlink_for_java_home
          fi
          ;;
        arm*)
          ;;
        ppc*)
          ;;
        *)
          create_symlink_for_java_home
          ;;
      esac
      ;;
    bsd)
      #  use bundled java if available
      case "${SYSTEM_ARCH}" in
        i[3-6]86)
          if [ -d "$CBS_HOME/java-freebsd-x86" ]; then
            JRE_HOME=$CBS_HOME/java-freebsd-x86
          else
            create_symlink_for_java_home
          fi
          ;;
        x86_64 | amd64)
          if [ -d "$CBS_HOME/java-freebsd-x64" ]; then
            JRE_HOME=$CBS_HOME/java-freebsd-x64
          else
            create_symlink_for_java_home
          fi
          ;;
        *)
          create_symlink_for_java_home
          ;;
      esac
      ;;
    *)
      create_symlink_for_java_home
      ;;
  esac
fi

# Assumed we always have this directory for JAVA_HOME and JRE_HOME at this stage
[ -d "$CBS_HOME/java" ] && JRE_HOME=$CBS_HOME/java
JAVA_HOME=$JRE_HOME
fi

##################################################################
# Define "JAVA limits" /"LD_LIBRARY_PATH" /"ROTATELOGS"

if [ -z "$JAVA_MEMORY_OPTS" ] || [ -z "$JAVA_EXTRA_OPTS" ]; then
SYSTEM_PHYSICAL_MEMORY_IN_KB=`get_physical_memory_in_kb`
SYSTEM_PHYSICAL_MEMORY_IN_MB=`expr $SYSTEM_PHYSICAL_MEMORY_IN_KB / 1024`

# Assumed we have 1GB memory for use in most case
SYSTEM_MEM_RESERVED_IN_MB=512
JAVA_THREAD_STACK_IN_KB=128
JAVA_HEAP_MIN_IN_MB=4

JAVA_EXTRA_OPTS="-verbose:gc -XX:+UseConcMarkSweepGC -XX:NewRatio=4 -XX:SurvivorRatio=4 -Xrs -XX:MaxMetaspaceSize=64m -XX:MinHeapFreeRatio=25 -XX:MaxHeapFreeRatio=50"

case "${SYSTEM_TYPE}" in
  # Modify JRE_HOME for Mac OS X
  # It is required to be here for this file will be used in upgrade patch as well

  linux)
    case "${SYSTEM_ARCH}" in
      i[3-6]86)
        JAVA_THREAD_STACK_IN_KB=256
        JAVA_HEAP_MAX_IN_MB=$JAVA64_HEAP_MAX_IN_MB
		LD_LIBRARY_PATH="$CBS_HOME/lib/LinX86"
		ROTATELOGS="$CBS_HOME/bin/LinX86/rotatelogs"
		export LD_LIBRARY_PATH ROTATELOGS
        ;;
	  x86_64 | amd64)
		JAVA_THREAD_STACK_IN_KB=256
        JAVA_HEAP_MAX_IN_MB=$JAVA64_HEAP_MAX_IN_MB
		LD_LIBRARY_PATH="$CBS_HOME/lib/LinX64"
		ROTATELOGS="$CBS_HOME/bin/LinX64/rotatelogs"
		export LD_LIBRARY_PATH ROTATELOGS
		;;
      arm*)
        JAVA_EXTRA_OPTS=""
        ;;
      ppc*)
        ;;
      *)
        ;;
    esac
    ;;
  bsd)
    case "${SYSTEM_ARCH}" in
	  i[3-6]86)
        JAVA_THREAD_STACK_IN_KB=256
        JAVA_HEAP_MAX_IN_MB=$JAVA64_HEAP_MAX_IN_MB
		LD_LIBRARY_PATH="$CBS_HOME/lib/FbdX86"
		ROTATELOGS="$CBS_HOME/bin/FbdX86/rotatelogs"
		export LD_LIBRARY_PATH ROTATELOGS
        ;;
	  x86_64 | amd64)
		JAVA_THREAD_STACK_IN_KB=256
        JAVA_HEAP_MAX_IN_MB=$JAVA64_HEAP_MAX_IN_MB
		LD_LIBRARY_PATH="$CBS_HOME/lib/FbdX64"
		ROTATELOGS="$CBS_HOME/bin/FbdX64/rotatelogs"
		export LD_LIBRARY_PATH ROTATELOGS
        ;;
      *)
        ;;
    esac
    ;;
  macos)
    case "${SYSTEM_ARCH}" in
      x86_64 | amd64)
        JAVA_THREAD_STACK_IN_KB=256
        JAVA_HEAP_MAX_IN_MB=$JAVA64_HEAP_MAX_IN_MB
	    LD_RUN_PATH="$CBS_HOME/lib/FbdX64"
		ROTATELOGS="$CBS_HOME/bin/FbdX64/rotatelogs"
		export LD_LIBRARY_PATH ROTATELOGS
        ;;
      *)
        SYSTEM_MEM_RESERVED_IN_MB=512
		LD_LIBRARY_PATH="$CBS_HOME/lib/FbdX86"
		ROTATELOGS="$CBS_HOME/bin/FbdX86/rotatelogs"
		export LD_LIBRARY_PATH ROTATELOGS
        ;;
    esac
    ;;
  solaris)
    SOLARIS_ARCH="`/usr/bin/isainfo -kv`"
    case "${SOLARIS_ARCH}" in
      64-bit*)
        JAVA_THREAD_STACK_IN_KB=256
        JAVA_HEAP_MAX_IN_MB=$JAVA64_HEAP_MAX_IN_MB
        JAVA_EXTRA_OPTS="-d64 ${JAVA_EXTRA_OPTS}"
        # set reserved memory here (64-bit)
        SYSTEM_MEM_RESERVED_IN_MB=1024
		LD_LIBRARY_PATH="$CBS_HOME/lib/SosX64"
		export LD_LIBRARY_PATH
		# Taks #10888
		# ROTATELOGS="$CBS_HOME/bin/SosX64/rotatelogs"
		# export LD_LIBRARY_PATH ROTATELOGS
        ;;
      *)
        JAVA_THREAD_STACK_IN_KB=128
        # set reserved memory here (32-bit)
        SYSTEM_MEM_RESERVED_IN_MB=512
		LD_LIBRARY_PATH="$CBS_HOME/lib/SosX86"
		export LD_LIBRARY_PATH ROTATELOGS
		# Taks #10888
		# ROTATELOGS="$CBS_HOME/bin/SosX86/rotatelogs"
		# export LD_LIBRARY_PATH ROTATELOGS
        ;;
    esac
    ;;
  *)
    ;;
esac

# It is assumed to have 4 cases of physical memory:
# 1) MEM <= 256MB
# 2) 256MB < MEM <= 512MB
# 3) 512MB < MEM <= 1GB
# 4) MEM > 1GB

if [ $SYSTEM_PHYSICAL_MEMORY_IN_MB -le 1024 ];then
  if [ $SYSTEM_PHYSICAL_MEMORY_IN_MB -le 512 ];then
    if [ $SYSTEM_PHYSICAL_MEMORY_IN_MB -le 256 ];then
      # 1) MEM <= 256MB
      SYSTEM_MEM_RESERVED_IN_MB=96
    else 
      # 2) 256MB < MEM <= 512MB
      SYSTEM_MEM_RESERVED_IN_MB=128
    fi
  else 
    # 3) 512MB < MEM <= 1GB
    SYSTEM_MEM_RESERVED_IN_MB=256
  fi
else
  # 4) MEM > 1GB
  SYSTEM_MEM_RESERVED_IN_MB=512
fi
# Server Options for process memory
# 
# Note: DON'T specify MAX_MEMORY with a value greater than the physical memory size

# Add special handling for UBS instances
if [ "${SYSTEM_IDENTITY}" = "UBS" ];then
  SYSTEM_PHYSICAL_MEMORY_IN_MB_HALVED=`expr $SYSTEM_PHYSICAL_MEMORY_IN_MB / 2`
  JAVA_THREAD_STACK_IN_KB=256
  JAVA_HEAP_MAX_IN_MB=$SYSTEM_PHYSICAL_MEMORY_IN_MB_HALVED
  SYSTEM_MEM_RESERVED_IN_MB=$SYSTEM_PHYSICAL_MEMORY_IN_MB_HALVED
fi

JAVA_XSS_VALUE="${JAVA_THREAD_STACK_IN_KB}"
JAVA_XMS_VALUE="256"
JAVA_XMX_VALUE=""


if [ $JAVA_HEAP_MAX_IN_MB -eq 0 ]; then
  # We cannot determine the physical memory, use empty JAVA_XMX_VALUE
  JAVA_XMX_VALUE=""
else
  if [ $SYSTEM_MEM_RESERVED_IN_MB -ge $SYSTEM_PHYSICAL_MEMORY_IN_MB ]; then
    # No more physical memory left for JAVA, use half of it
    JAVA_XMX_VALUE=`expr $SYSTEM_PHYSICAL_MEMORY_IN_MB / 2`
  else
    SYSTEM_AVAILABLE_MEMORY=`expr $SYSTEM_PHYSICAL_MEMORY_IN_MB - $SYSTEM_MEM_RESERVED_IN_MB`
    if [ $JAVA_HEAP_MAX_IN_MB -ge $SYSTEM_AVAILABLE_MEMORY ]; then
      JAVA_XMX_VALUE="$SYSTEM_AVAILABLE_MEMORY"
    else
      JAVA_XMX_VALUE="$JAVA_HEAP_MAX_IN_MB"
    fi
  fi
fi

# Use the minimum supported value for Xmx if it is too small
[ $JAVA_HEAP_MIN_IN_MB -ge $JAVA_XMX_VALUE ] && JAVA_XMX_VALUE="$JAVA_HEAP_MIN_IN_MB"

# Use same value for the -Xms and -Xmx option
[ $JAVA_HEAP_MIN_MAX_SAME -eq 1 ] && JAVA_XMS_VALUE="$JAVA_XMX_VALUE"

# Now we have the suitable java memory options
JAVA_MEMORY_OPTS="-Xss${JAVA_XSS_VALUE}k"
[ ! -z "${JAVA_XMS_VALUE}" ] && JAVA_MEMORY_OPTS="${JAVA_MEMORY_OPTS} -Xms${JAVA_XMS_VALUE}m"
[ ! -z "${JAVA_XMX_VALUE}" ] && JAVA_MEMORY_OPTS="${JAVA_MEMORY_OPTS} -Xmx${JAVA_XMX_VALUE}m"
# if Could not reserve enough space for object heap, uncomment the following
# JAVA_MEMORY_OPTS="-Xss256k -Xms256m -Xmx256m"
fi

CLASSPATH="$CBS_HOME/lib/*:$CBS_HOME/tomcat/lib/*"
JAVA_MEMORY_OPTS="$VM_OPTS -Xms128m -Xmx512m"
JAVA_MEMORY_OPTS="$VM_OPTS -Xms128m -Xmx512m"
JAVA_EXTRA_OPTS="-Xrs -XX:+UseConcMarkSweepGC -XX:+UseParNewGC -XX:NewRatio=4 -XX:SurvivorRatio=6 -XX:MetaspaceSize=128m"
JAVA_NETWORK_OPTS="-Dsun.net.inetaddr.ttl=10800 -Dnetworkaddress.cache.ttl=10800 -Dsun.net.inetaddr.negative.ttl=0 -Dnetworkaddress.cache.negative.ttl=0 -Djava.net.preferIPv4Stack=true"
JAVA_JNI_OPTS="-Djava.library.path=$LD_LIBRARY_PATH"
MAIN_CLASS=MigrateV7

V7_CBS_HOME=$1

echo "$V7_CBS_HOME"

if [ "$V7_CBS_HOME" = "" ]; then
   ERR_MSG="*** Please provide a V7_CBS_HOME parameter!"

fi

echo "============= $0 Parameters ============="
echo "CBS_HOME:            $V7_CBS_HOME"
echo "=========================================================="
echo "$CLASSPATH"
echo "$MAIN_CLASS"
$JAVA_HOME/bin/java "$JAVA_EXTRA_OPTS" -cp "$CLASSPATH" "$MAIN_CLASS" "$V7_CBS_HOME" 2>&1
