#!/bin/sh

# ------------------------- Retrieve APP_HOME ----------------------------------
cd `dirname "$0"`
APP_BIN=`pwd`
APP_HOME=`dirname "$APP_BIN"`
UTIL_HOME="${APP_HOME}/util"
FIX_PERMISSION_OPTION="FIX_PERMISSION"
FORCE_INHERIT_TAG="FORCE_INHERIT"

SYSTEM_OS=`uname`
SYSTEM_TYPE=
case "${SYSTEM_OS}" in
  Linux*)
    SYSTEM_TYPE="linux"
    ;;
  Solaris*)
    SYSTEM_TYPE="solaris"
    ;;
  SunOS*)
    SYSTEM_TYPE="solaris_sunos"
    ;;
  *BSD*)
    SYSTEM_TYPE="bsd"
    ;;
  **)
    echo "This OS ${SYSTEM_OS} is not supported by this script! Exit `basename $0` now!"
    exit 1
    ;;
esac

# ------------------------- Fix permission if mode set to FIX_PERMISSION ----------------------------------
getFileInfo()
{
    RESULT_FILE_OWNER_AND_GROUP="$1"
    RESULT_FILE_MODE="$2"
    FILE_PATH="$3"

    eval $RESULT_FILE_OWNER_AND_GROUP=""
    eval $RESULT_FILE_MODE=""

    case "${SYSTEM_TYPE}" in
        solaris*)
          FILE_INFO=`ls -ld "${FILE_PATH}" | nawk '{k=0;for(i=0;i<=8;i++)k+=((substr($1,i+2,1)~/[rwx]/)*2^(8-i));if(k)printf("%0o ",k);print $3":"$4}'`
          ;;
        bsd*)
          FILE_INFO=`stat -f "%0Lp %Su:%Sg" "${FILE_PATH}"`
          ;;
        **)
          FILE_INFO=`stat -c "%a %U:%G" "${FILE_PATH}"`
          ;;
    esac
    MODE=`echo "${FILE_INFO}" | cut -d ' ' -f 1`
    OWNER_AND_GROUP=`echo "${FILE_INFO}" | cut -d ' ' -f 2`
    if [ ! -z "${OWNER_AND_GROUP}" ]; then
        eval $RESULT_FILE_OWNER_AND_GROUP="${OWNER_AND_GROUP}"
    fi
    if [ ! -z "${MODE}" ]; then
        eval $RESULT_FILE_MODE="${MODE}"
    fi
}

inheritPermissionAndOwner() {
    TARGET_PATH="$1"
    FORCE_INHERIT_OPTION="$2"
    echo Start checking permission of "${TARGET_PATH}"
    if [ ! -d "${TARGET_PATH}" ]; then
        echo Directory "${TARGET_PATH}" does not exist
        return
    fi
    PARENT_PATH=`dirname "$1"`
    PARENT_OWNER_AND_GROUP=""
    PARENT_MODE=""
    getFileInfo PARENT_OWNER_AND_GROUP PARENT_MODE "${PARENT_PATH}"

    # when force inherit, skip getting current file info to fail the checking
    FILE_OWNER_AND_GROUP=""
    FILE_MODE=""
    if [ "${FORCE_INHERIT_TAG}" != "${FORCE_INHERIT_OPTION}" ]; then
        getFileInfo FILE_OWNER_AND_GROUP FILE_MODE "${TARGET_PATH}"
    fi

    # fix unmatched file mode
    if [ -z "${PARENT_MODE}" ]; then
        echo Skip fixing file mode "${TARGET_PATH}" because parent mode cannot be acquired
    elif [ "${FILE_MODE}" != "${PARENT_MODE}" ]; then
        echo Inherit parent mode "${PARENT_MODE}" to "${TARGET_PATH}"
        chmod -R "${PARENT_MODE}" "${TARGET_PATH}"
    else
        echo Skip fixing file mode "${TARGET_PATH}" because parent mode "${PARENT_MODE}" match with current mode "${FILE_MODE}"
    fi

    # fix unmatched owner and group
    if [ -z "${PARENT_OWNER_AND_GROUP}" ];then
        echo Skip fixing file owner "${TARGET_PATH}" because parent owner cannot be acquired
    elif [ "${FILE_OWNER_AND_GROUP}" != "${PARENT_OWNER_AND_GROUP}" ]; then
        echo Inherit parent owner and group "${PARENT_OWNER_AND_GROUP}" to "${TARGET_PATH}"
        chown -R "${PARENT_OWNER_AND_GROUP}" "${TARGET_PATH}"
    else
        echo Skip fixing file owner "${TARGET_PATH}" because parent owner "${PARENT_OWNER_AND_GROUP}" match with current owner "${FILE_OWNER_AND_GROUP}"
    fi
}

# Fix File Permission only
if [ "${FIX_PERMISSION_OPTION}" = "$1" ]; then
    echo "Fix File Permissions"
    # jvm permission may become wrong again when update
    inheritPermissionAndOwner "${APP_HOME}/jvm" "${FORCE_INHERIT_TAG}"
    inheritPermissionAndOwner "${APP_HOME}/ipc"
    inheritPermissionAndOwner "${APP_HOME}/log"

    for SETTING_HOME in `cat ${APP_HOME}/home.txt`; do
        if [ -d "$SETTING_HOME" ]; then
            # handle log folder could be 777 case
            # for BACKUP_SET_FOLDER in `ls ${SETTING_HOME}/log`; do
            #     inheritPermissionAndOwner "${SETTING_HOME}/log/${BACKUP_SET_FOLDER}/Backup"
            # done
            # inheritPermissionAndOwner "${SETTING_HOME}/ipc/BackupSet"
            chmod -R 755 "${SETTING_HOME}/log"
            chmod -R 755 "${SETTING_HOME}/ipc"
        fi
    done
    echo "Fix File Permissions End"
    exit 0
fi

# -------------------- Print Logging Message Header ----------------------------
echo "Log Time: `date`"

# ------------ Verify if the privilege is enough for install ------------------
## Verify the privilege if the shell script "privilege.sh" exist.
if [ -z "$1" ]; then
echo ""
if [ -f "$UTIL_HOME/bin/privilege.sh" ]
then
  echo "Verifying current user privilege ..."
  "$UTIL_HOME/bin/privilege.sh" "config"
  [ $? -ne 0 ] && echo "Exit \"`basename $0`\" now!" && exit 1
else
  echo "The shell script \"$UTIL_HOME/bin/privilege.sh\" is missing."
  echo "Exit \"`basename $0`\" now!" && exit 1
fi
echo "Current user has enough privilege to \"config\"."
echo ""
fi

# -------------------- Print Logging Message  ----------------------------------

OS_IS_LINUX=0

case "${SYSTEM_TYPE}" in
  linux)
    echo "Start configuration on Generic Linux Platform (`uname`)"
    OS_IS_LINUX=1
    ;;
  solaris)
    echo "Start configuration on Solaris 2.X Platform (`uname`)"
    ;;
  solaris_sunos)
    echo "Start configuration on Solaris 5.X Platform (`uname`)"
    ;;
  bsd)
    echo "Start configuration on BSD distribution Platform (`uname`)"
    ;;
esac

echo ""
echo "Installation Path: ${APP_HOME}"

# ----------------------- Configure Application --------------------------------
# Get the JAVA Home path.
cd "${APP_HOME}"

# Verify the JAVA_EXE whether it is a valid JAVA Executable or not.
STRING_JAVA_VERSION="java version,openjdk version"
OUTPUT_JAVA_VERSION=`"${APP_HOME}/jvm/bin/java" -version 2>&1`
OUTPUT_JVM_SUPPORT=0
BACKUP_IFS=$IFS
IFS=","
for word in $STRING_JAVA_VERSION; do
    if [ `echo "${OUTPUT_JAVA_VERSION}" | grep "${word}" | grep -cv "grep ${word}"` -le 0 ]
    then
      #echo "The Java Executable \"${APP_HOME}/jvm/bin/java\" is not a valid Java Executable. Exit \""`basename "$0"`"\" now."
      continue;
    else
      OUTPUT_JVM_SUPPORT=1
      break;
    fi
done
IFS=$BACKUP_IFS
if [ $OUTPUT_JVM_SUPPORT -eq 0 ]
then
    echo "The Java Executable \"${APP_HOME}/jvm/bin/java\" is not a valid Java Executable. Exit \""`basename "$0"`"\" now."
    exit 1
fi

# Verify if the JVM version in the JVM Home are supported
MINIMUM_SUPPORTED_JVM_VERSION=1.7 # The JVM supported Version is defined in APP v7.0 onwards.
MAXIMUM_SUPPORTED_JVM_VERSION=1.8 # The JVM supported Version is defined in APP v7.0 onwards.
echo "Minimum supported JVM version: $MINIMUM_SUPPORTED_JVM_VERSION"
echo "Maximum supported JVM version: $MAXIMUM_SUPPORTED_JVM_VERSION"
[ ! -f "$UTIL_HOME/bin/verify-jvm-version.sh" ] && echo "The shell script \"$UTIL_HOME/bin/verify-jvm-version.sh\" is missing." && echo "Exit \"`basename $0`\" now!" && exit 1
"$UTIL_HOME/bin/verify-jvm-version.sh" "$APP_HOME/jvm" "$MINIMUM_SUPPORTED_JVM_VERSION" "$MAXIMUM_SUPPORTED_JVM_VERSION" 1>"/dev/null" 2>&1
if [ $? -ne 0 ]
then
    [ -L "$APP_HOME/jvm" ] && rm -f "$APP_HOME/jvm" && echo "Removed the Symlink \"$APP_HOME/jvm\"."
    echo "The JVM version is out of range \"$MINIMUM_SUPPORTED_JVM_VERSION\" - \"$MAXIMUM_SUPPORTED_JVM_VERSION\" which is not supported by the APP."
    echo "Please change the JAVA_HOME Directory and run the installation again."
    echo "Exit \"`basename $0`\" now!"
    exit 1
fi

echo "Current JVM version is supported for installation."
if [ ! -x "$APP_HOME/jvm/bin/bJW" ]; then
    echo "Create Backup Manager JVM, Path: $APP_HOME/jvm/bin/bJW"
    ln -sf "$APP_HOME/jvm/bin/java" "$APP_HOME/jvm/bin/bJW"
    chmod 755 "$APP_HOME/jvm/bin/bJW"
fi

if [ ! -x "$APP_HOME/jvm/bin/bschJW" ]; then
    echo "Create Scheduler Service JVM, Path: $APP_HOME/jvm/bin/bschJW"
    ln -sf "$APP_HOME/jvm/bin/java" "$APP_HOME/jvm/bin/bschJW"
    chmod 755 "$APP_HOME/jvm/bin/bschJW"
fi

# Set File Permission
echo "Setup File Permissions"
touch "$APP_HOME/home.txt"
chmod 777 "$APP_HOME/home.txt"
chmod 755 $APP_BIN/*
chmod 777 "$APP_BIN/notesenv"
chmod 755 $UTIL_HOME/*

exit 0
