#!/bin/sh

# Check valid URL path
# 1 - Return Result
# 2 - URL
checkValidUrl()
{
    RESULT_VALID_URL="$1"
    URL=`echo "$2" | tr [:upper:] [:lower:]`

    if [ `echo $URL | cut -c1-5` = "http:" -o `echo $URL | cut -c1-6` = "https:" ]; then
        eval $RESULT_VALID_URL="Y"
    else
        eval $RESULT_VALID_URL="N"
    fi
}

# Extract server address from name in format obm-mac-%PORT%-%HOST_NAME%-%PROTOCOL%-%OWNER_ID%.sh, where %PROTOCOL% is "http"/"https"
# 1 - Return HostAddr
# 2 - Return OwnerID
# 3 - Name
parseHostAddrFromName()
{
    RESULT_HOST_ADDR="$1"
    RESULT_OWNER_ID="$2"
    # add prefix here to avoid failure case "-abc", prefix will be abandoned below
    NAME=`basename "prefix$3"`

    eval $RESULT_HOST_ADDR=""
    eval $RESULT_OWNER_ID=""

    PROTOCOL=""
    SUFFIX_NAME=""
    # remove file ext
    NAME=`echo "$NAME" | sed "s|\.[^\.]*$||"`

    while true; do
        # sed "s|^[^-]*$||": replace whole string with empty if whole string is without '-'
        if [ -z "`echo $NAME | sed \"s|^[^-]*$||\"`" ]; then
            break;
        fi
        # get suffix
        SUFFIX=`echo "$NAME" | sed "s|^.*-||"`
        # remove suffix
        NAME=`echo "$NAME" | sed "s|-[^-]*$||"`
        if [ ! -z "`echo $SUFFIX | grep \"^https\"`" ]; then
            PROTOCOL="https"
            break;
        fi
        if [ ! -z "`echo $SUFFIX | grep \"^http\"`" ]; then
            PROTOCOL="http"
            break;
        fi
        if [ -z "$SUFFIX_NAME" ]; then
            SUFFIX_NAME="$SUFFIX"
        else
            SUFFIX_NAME="$SUFFIX-$SUFFIX_NAME"
        fi
    done

    if [ ! -z "$PROTOCOL" ]; then
        # remove prefix
        NAME=`echo "$NAME" | sed "s|[^-]*-||"`
        while true; do
            # sed "s|^[^-]*$||": replace whole string with empty if whole string is without '-'
            if [ -z "`echo $NAME | sed \"s|^[^-]*$||\"`" ]; then
                break;
            fi
            # PORT must be either a number or empty
            PORT=`echo "$NAME" | cut -d'-' -f1 | sed "s|.*[^0-9].*||"`
            NAME=`echo "$NAME" | sed "s|[^-]*-||"`
            # check if PORT is valid
            if [ ! -z "$PORT" ] && [ "$PORT" -le 65535 ] && [ "$PORT" -ge 0 ]; then
                if [ ! -z "$NAME" ]; then
                    eval "$RESULT_HOST_ADDR=\"$PROTOCOL://$NAME:$PORT\""
                fi
                break;
            fi
        done
    fi
    if [ ! -z "$SUFFIX_NAME" ]; then
        # remove character 'O' and string following it
        # this is not number "zero", it is a upper case 'o'
        SUFFIX_NAME=`echo "$SUFFIX_NAME" | sed "s|O[^O]*$||"`
        eval "$RESULT_OWNER_ID=\"$SUFFIX_NAME\""
    fi
}

# Download and setup prepackaged components
# 1 - Return Result
# 2 - URL
# 3 - File Name
# 4 - URL Argu
# 5 - Destination
# 6 - Package Type
downloadAndSetupComponent()
{
    SETUP_RESULT="$1"
    SOURCE_URL_HOME="$2"
    SOURCE_FILE="$3"
    SOURCE_URL_ARGU="$4"
    DESTINATION_DIR="$5"
    PKG_TYPE="$6"

    if [ ! -z "$SOURCE_URL_ARGU" ]; then
        DOWNLOAD_URL="$SOURCE_URL_HOME/$SOURCE_FILE?$SOURCE_URL_ARGU"
    else
        DOWNLOAD_URL="$SOURCE_URL_HOME/$SOURCE_FILE"
    fi
    CURRENT_DIR=`pwd`
    if [ "$PKG_TYPE" = "jvm" ] ; then
        TARGET_DIR="$DESTINATION_DIR/jvm"
    else
        TARGET_DIR="$DESTINATION_DIR"
    fi

    echo "Downloading file... $SOURCE_FILE"
    curl -k "$DOWNLOAD_URL" -o "$SOURCE_FILE"
    if [ -f "$SOURCE_FILE" ]; then
        echo "Download file completed"
        if [ ! -d "$TARGET_DIR" ]; then
            mkdir "$TARGET_DIR"
        fi
        echo "Untar component file to $TARGET_DIR"
        tar -xzf "$SOURCE_FILE" -C "$TARGET_DIR" > /dev/null
        rm "$SOURCE_FILE"
        eval $SETUP_RESULT="Y"
    else
        echo "Fail to download file from URL $DOWNLOAD_URL"
        eval $SETUP_RESULT="N"
    fi
}

# Verify and setup prepackaged components
# 1 - Return Result
# 2 - Source
# 3 - File Name
# 4 - Destination
# 5 - Package Type
verifyAndSetupComponent()
{
    SETUP_RESULT="$1"
    SOURCE_DIR="$2"
    SOURCE_FILE="$3"
    DESTINATION_DIR="$4"
    PKG_TYPE="$5"

    CURRENT_DIR=`pwd`
    if [ "$PKG_TYPE" = "jvm" ] ; then
        TARGET_DIR="$DESTINATION_DIR/jvm"
    else
        TARGET_DIR="$DESTINATION_DIR"
    fi

    if [ -f "$SOURCE_DIR/$SOURCE_FILE" ]; then
        if [ ! -d "$TARGET_DIR" ]; then
            mkdir "$TARGET_DIR"
        fi
        echo "Untar $SOURCE_FILE to $TARGET_DIR"
        tar -xzf "$SOURCE_DIR/$SOURCE_FILE" -C "$TARGET_DIR" > /dev/null
        eval $SETUP_RESULT="Y"
    else
        echo "$SOURCE_FILE not exists"
        eval $SETUP_RESULT="N"
    fi
}

# Prepare prepackaged components
# 1 - Return Result
preparePackagedComponents()
{
    PREPARE_RESULT="$1"
    eval $PREPARE_RESULT="N"

    URL_ARGU=""
    HOST_ADDR=""
    OWNER_ID=""

    DOWNLOAD_PGK="Y"
    # Parse host address
    parseHostAddrFromName HOST_ADDR OWNER_ID "$RESTORER"
    if [ -z "$HOST_ADDR" ] && [ -d "$CLIENT_VERSION_PKG_DIR" ]; then
        echo "Using package in $CLIENT_VERSION_PKG_DIR"
        DOWNLOAD_PGK="N"
    fi

    if [ "$DOWNLOAD_PGK" = "N" ] ; then
        if [ ! -f "$PKG_DIR/version.txt" ]; then
            echo "Package version file not exists"
            return
        fi
        PKG_VERSION=`cat "$PKG_DIR/version.txt"`
        echo "Package version: $PKG_VERSION"
        MIN_VERSION="$(printf "$VERSION\n$PKG_VERSION\n" | sort -t. -k 1,1n -k 2,2n -k 3,3n -k 4,4n | head -n1)"
        if [ "$MIN_VERSION" = "$VERSION" ] && [ "$PKG_VERSION" != "$VERSION" ]; then
            echo "Package version does not match with installer version: $VERSION"
            return
        fi
    else
        if [ -z "$HOST_ADDR" ]; then
            echo "Host address is not defined"
            return
        fi
        if [ -z "$OWNER_ID" ]; then
            echo "Owner ID is not defined"
            return
        fi
        IS_VALID_URL="N"
        checkValidUrl IS_VALID_URL "$HOST_ADDR"
        if [ "$IS_VALID_URL" = "N" ]; then
            echo "Invalid host address - $HOST_ADDR"
            return
        fi
        echo "Host address: $HOST_ADDR"

        # Check jvm url
        if [ ! -z "$JVM_URL" ] ; then
            IS_VALID_URL="N"
            checkValidUrl IS_VALID_URL "$JVM_URL"
            if [ "$IS_VALID_URL" = "N" ]; then
                JVM_URL="$HOST_ADDR/$JVM_URL"
            fi
        fi
        if [ -z "$JVM_URL" ] ; then
            echo "JVM package's URL is not defined"
            return
        fi

        # Check component url
        if [ ! -z "$COMPONENT_URL" ] ; then
            IS_VALID_URL="N"
            checkValidUrl IS_VALID_URL "$COMPONENT_URL"
            if [ "$IS_VALID_URL" = "N" ]; then
                COMPONENT_URL="$HOST_ADDR/$COMPONENT_URL"
            fi
        fi
        if [ -z "$COMPONENT_URL" ] ; then
            echo "Component package's URL is not defined"
            return
        fi

        URL_VERSION_ARGU_TAG="version"
        URL_OWNER_ID_ARGU_TAG="ownerid"
        if [ "$URL_VERSION_ARGU_TAG" != "" ] && [ "$URL_OWNER_ID_ARGU_TAG" != "" ]; then
          URL_ARGU="$URL_VERSION_ARGU_TAG=$VERSION&$URL_OWNER_ID_ARGU_TAG=$OWNER_ID"
        elif [ "$URL_OWNER_ID_ARGU_TAG" != "" ]; then
          URL_ARGU="$URL_OWNER_ID_ARGU_TAG=$OWNER_ID"
        elif [ "$URL_VERSION_ARGU_TAG" != "" ]; then
          URL_ARGU="$URL_VERSION_ARGU_TAG=$VERSION"
        fi
    fi

    # Prepare jvm
    PKG_JVM_FILE="jre-std-mac.tar.gz"
    if [ -z "$PKG_JVM_FILE" ]; then
        echo "No JVM package is defined"
    else
        for word in $PKG_JVM_FILE
        do
            if [ "$DOWNLOAD_PGK" = "N" ] ; then
                verifyAndSetupComponent SETUP_RESULT "$CLIENT_VERSION_PKG_DIR" "$word" "$APP_TMP" "jvm"
            else
                downloadAndSetupComponent SETUP_RESULT "$JVM_URL" "$word" "$URL_ARGU" "$APP_TMP" "jvm"
            fi
            if [ "$SETUP_RESULT" = "N" ] ; then
                echo "Failed to prepare JVM files."
                return
            fi
        done
    fi

    # Prepare other components
    PKG_COMPONENT_FILE="app-common.tar.gz app-mac-obm.tar.gz app-native-mac.tar.gz aua-common.tar.gz aua-mac-obm.tar.gz aua-native-mac.tar.gz util-common.tar.gz util-mac-obm.tar.gz properties-common.tar.gz app-inst-mac-obm.tar.gz aua-inst-mac-obm.tar.gz app-custom-common-obm.tar.gz"
    if [ -z "$PKG_COMPONENT_FILE" ]; then
        echo "No component package is defined"
    else
        for word in $PKG_COMPONENT_FILE
        do
            if [ "$DOWNLOAD_PGK" = "N" ] ; then
                verifyAndSetupComponent SETUP_RESULT "$CLIENT_VERSION_PKG_DIR" "$word" "$APP_TMP" "others"
            else
                downloadAndSetupComponent SETUP_RESULT "$COMPONENT_URL" "$word" "$URL_ARGU" "$APP_TMP" "others"
            fi
            if [ "$SETUP_RESULT" = "N" ] ; then
                echo "Failed to prepare component files."
                return
            fi
        done
    fi

    # Changed to not remove offline installer package
    # if [ "$DOWNLOAD_PGK" = "N" ] ; then
    #     rm -rf "$PKG_DIR"
    # fi
    eval $PREPARE_RESULT="Y"
}

# ---------------------------------
# Initialize
cd "`dirname "$0"`"

RESTORER="$0"
EXEC_HOME=`pwd`
APP_HOME=$EXEC_HOME/RestoreWizard
APP_BIN=$APP_HOME/bin
TIMESTAMP_DIR=`date '+%y%m%d%H%M%S'`
APP_TMP="/tmp/_obm.$TIMESTAMP_DIR"
PKG_DIR="$EXEC_HOME/app.pkg"

# Print Logging Message Header
echo "Log Time: `date`"

# Determine the OS information
OS_VERSION=$(sw_vers -productVersion | awk -F '.' '{print $1}')
OS_NAME=$(sw_vers -productName)
ARCH=$(uname -m)
echo "OS Info: $OS_NAME $OS_VERSION $ARCH"

PRODUCT_NAME="AhsayOBM"

# Define URLs of client versions
OLD_JVM_URL="obs/download/9/jvm"
OLD_COMPONENT_URL="obs/download/9/component"
FALLBACK_MAJOR_VERSION="9"

NEW_JVM_URL="obs/download/10/jvm"
NEW_COMPONENT_URL="obs/download/10/component"
VERSION="10.1.4.0"
NEW_CLIENT_MAJOR_VERSION=$(echo "$VERSION" | cut -d'.' -f1)

# Decide which client version to use
CLIENT_MAJOR_VERSION="$FALLBACK_MAJOR_VERSION"
# Check if macOS version is within the range and if it's 64-bit
if [[ "$OS_VERSION" -ge 11 && "$OS_VERSION" -le 15 ]] && [[ "$ARCH" == "x86_64" || "$ARCH" == "arm64" ]]; then
    CLIENT_MAJOR_VERSION="$NEW_CLIENT_MAJOR_VERSION"
fi

CLIENT_VERSION_PKG_DIR="$PKG_DIR/$CLIENT_MAJOR_VERSION"
if [ "$CLIENT_MAJOR_VERSION" = "$FALLBACK_MAJOR_VERSION" ]; then
    JVM_URL="$OLD_JVM_URL"
    COMPONENT_URL="$OLD_COMPONENT_URL"
    echo "$PRODUCT_NAME v$NEW_CLIENT_MAJOR_VERSION does not support this operating system version of your machine. Installing $PRODUCT_NAME v$CLIENT_MAJOR_VERSION instead."
    osascript -e "display dialog \"$PRODUCT_NAME v$NEW_CLIENT_MAJOR_VERSION does not support this operating system version of your machine. Installing $PRODUCT_NAME v$CLIENT_MAJOR_VERSION instead.\" buttons {\"OK\"} default button \"OK\" with title \"Install $PRODUCT_NAME v$CLIENT_MAJOR_VERSION\""
else
    JVM_URL="$NEW_JVM_URL"
    COMPONENT_URL="$NEW_COMPONENT_URL"
fi

# Prepare packaged components
if [ -d "$APP_TMP" ]; then
    rm -rf "$APP_TMP"
fi
mkdir "$APP_TMP"

preparePackagedComponents RESULT
if [ "$RESULT" = "N" ] ; then
    echo "Error in preparing files, installer will terminate"
    rm -rf "$APP_TMP"
    exit 1
fi

# Move packaged components
if [ ! -d "$APP_HOME" ]; then
    mkdir "$APP_HOME"
fi
mv "$APP_TMP"/* "$APP_HOME"
rm -rf "$APP_TMP"

if [ "$DOWNLOAD_PGK" = "N" ] ; then
    echo "Fixing extended attributes of files in $APP_HOME."
    xattr -rd com.apple.quarantine "$APP_HOME" 2>/dev/null
fi

# Do post processing
JAVA_HOME="$APP_HOME/jvm"
JAVA_EXE="$JAVA_HOME/bin/java"
LD_LIBRARY_PATH="$APP_HOME/bin:$LD_LIBRARY_PATH"
SHLIB_PATH="$LD_LIBRARY_PATH"
export LD_LIBRARY_PATH SHLIB_PATH

# Change to APP_BIN for JAVA execution
cd "${APP_BIN}"

# Reference path will be used to avoid empty space in the parent directory
LIB_HOME=.
JAVA_OPTS=$(awk 'NF {printf "%s%s", sep, $0; sep=" "} END {print ""}' cb.ini)
JNI_PATH="-Djava.library.path=$LIB_HOME:$LIB_HOME/MacX84"
CLASSPATH="$LIB_HOME:$LIB_HOME/cb.jar"
MAIN_CLASS=InstallerUtil

INSTALLER_MODE="-i"
PRODUCT_EDITION="obm"

echo "Running installer post processes."

# Do not include double-quote for java options, jni path, classpath,
# main class, installer mode, and product edition
# Only apply double-quote for app home
"${JAVA_EXE}" $JAVA_OPTS $JNI_PATH -cp $CLASSPATH $MAIN_CLASS $INSTALLER_MODE $PRODUCT_EDITION "${APP_HOME}"

# Verify if run script exist
RUN_SCRIPT="$APP_BIN/RunRestore.sh"
if [ ! -f "$RUN_SCRIPT" ]; then
    echo "Run script not exist"
    rm -rf "$APP_HOME"
    exit 1
fi
osascript -e "do shell script \"${RUN_SCRIPT}\""

# Exit
cd "$EXEC_HOME"
# Skip removal for granting permission
# rm -rf "$APP_HOME"
echo "Done"
exit 0
