#!/bin/sh
# [Ahsay Cloud Backup Suite] (version 10.1.4.0)

# -------------------------- Retrieve CBS_HOME ---------------------------------
cd `dirname "$0"`
CBS_BIN=`pwd`
CBS_HOME=`dirname $CBS_BIN`
CBS_UTIL_BIN=$CBS_HOME/util/bin

# ---------------------- Print Logging Message Header --------------------------
echo "Log Time: `date`"

# ------------ Verify if the privilege is enough for install ------------------
# Comment on 30-Aug-2010:
## Verify the privilege if the shell script "privilege.sh" exist. (which is aligned with the OBC)
echo ""
if [ -f "$CBS_UTIL_BIN/privilege.sh" ]
then
  echo "Verifying current user privilege ..."
  "$CBS_UTIL_BIN/privilege.sh" "install"
  [ $? -ne 0 ] && echo "Exit \"`basename $0`\" now!" && exit 1
else
  echo "The shell script \"$CBS_UTIL_BIN/privilege.sh\" is missing."
  echo "Exit \"`basename $0`\" now!" && exit 1
fi
echo "Current user has enough privilege to \"install\"."
echo ""

# -------------------- Print Logging Message -----------------------------------
OS_IS_LINUX=0
OS_IS_FREEBSD=0
case `uname` in
    Linux*)
        echo "Start configuration on Generic Linux Platform (`uname`)"
        OS_IS_LINUX=1
    ;;
    Solaris*) echo "Start configuration on Solaris 2.X Platform (`uname`)";;
    SunOS*)   echo "Start configuration on Solaris 5.X Platform (`uname`)";;
    OpenBSD*)
        echo "This OS `uname` is not supported by this script! Exit `basename $0` now!"
        exit 1
    ;;
    *BSD*)
        echo "Start configuration on BSD distribution Platform (`uname`)"
        OS_IS_FREEBSD=1
    ;;
    **)
        echo "This OS `uname` is not supported by this script! Exit `basename $0` now!"
        exit 1
    ;;
esac

# ------------------------ Check if OS is supported --------------------------
IS_OS_SUPPORTED=0
if [ "$OS_IS_LINUX" = "1" ]; then
    MINIMUM_SUPPORTED_KERNEL_VERSION="4.0"
    KERNEL_VERSION=$(uname -r | cut -d- -f1)  # Get only version part, strip any suffix
    ARCH_RAW=$(uname -m)
    if { [ "$ARCH_RAW" = "x86_64" ] || [ "$ARCH_RAW" = "amd64" ] || [ "$ARCH_RAW" = "aarch64" ] || [ "$ARCH_RAW" = "arm64" ]; } && \
            [ "$(printf '%s\n' "$MINIMUM_SUPPORTED_KERNEL_VERSION" "$KERNEL_VERSION" | sort -V | head -n1)" = "$MINIMUM_SUPPORTED_KERNEL_VERSION" ]; then
        IS_OS_SUPPORTED=1
    fi
elif [ "$OS_IS_FREEBSD" = "1" ]; then
    MINIMUM_SUPPORTED_BSD_VERSION="13.0"
    BSD_VERSION=$(freebsd-version | cut -d- -f1)   # Sample result: 14.3-RELEASE, become 14.3 after cut
    ARCH_RAW=$(uname -m)
    # compare by sorting version with min version in ascending order and check if the first one is min version or not, similar to version >= min version
    if { [ "$ARCH_RAW" = "x86_64" ] || [ "$ARCH_RAW" = "amd64" ]; } && \
            [ "$(printf '%s\n' "$MINIMUM_SUPPORTED_BSD_VERSION" "$BSD_VERSION" | sort -V | head -n1)" = "$MINIMUM_SUPPORTED_BSD_VERSION" ]; then
        IS_OS_SUPPORTED=1
    fi
fi

if [ "$IS_OS_SUPPORTED" = "0" ]; then
    PRODUCT_NAME="AhsayCBS"
    VERSION="10.1.4.0"
    NEW_CLIENT_MAJOR_VERSION=$(echo "$VERSION" | cut -d'.' -f1)
    echo "$PRODUCT_NAME v$NEW_CLIENT_MAJOR_VERSION does not support this operating system version of your machine. Please use $PRODUCT_NAME v9 instead." && exit 1
fi

# ---------------------- Setup CBS required parameters -------------------------

# Current directory must be ${CBS_HOME} for java symbolic link
cd "${CBS_HOME}"
BUNDLED_JAVA_HOME=java
echo "Using CBS_HOME $CBS_HOME"

# [Start] 41483: Starting v10, java 21 will be used
JAVA_DIR="java21-linux-x64"
# The JVM supported version starting v10
JAVA_VERSION="21.0"
# [End] 41483

setup_custom_jvm() {
    JAVA_USER=jvm
    # 41483: Starting v10, java 21 will be used
    # echo "Please enter your java 1.8 home: "
    echo "Please enter your java $JAVA_VERSION home: "
    read USER_JVM_HOME
    if [ -z "$USER_JVM_HOME" ]; then
        echo "Invalid java home!"
        exit 1
    fi
    if [ ! -d "$USER_JVM_HOME" ]; then
        echo "$USER_JVM_HOME does not exist!"
        exit 1
    fi
    # 41483: Starting v10, java 21 will be used
    # echo "Copy java 1.8 from $USER_JVM_HOME"
    echo "Copy java $JAVA_VERSION from $USER_JVM_HOME"
    if [ ! -d "$JAVA_USER" ]; then
        cp -rf "$USER_JVM_HOME" "$JAVA_USER"
    else
        cp -rf $USER_JVM_HOME/* $JAVA_USER
    fi
    JVM_SECURITY="${CBS_HOME}/util/jvm/lib/security"
    if [ -d "$JVM_SECURITY" ]; then
        cp -rf $JVM_SECURITY/* $JAVA_USER/lib/security
    else
        # 41483: Starting v10, java 21 will be used
        # JVM_SECURITY="${CBS_HOME}/java-linux-x64/lib/security"
        JVM_SECURITY="${CBS_HOME}/${JAVA_DIR}/lib/security"
        if [ -d "$JVM_SECURITY" ]; then
            cp $JVM_SECURITY/cacerts $JAVA_USER/lib/security
        fi
    fi
    BUNDLED_JAVA_HOME=$JAVA_USER
}

# 22765: Verify if GNU Libc version is supported for Linux
# 26508: Support Debian
# DIST=`grep DISTRIB_ID /etc/*-release | awk -F '=' '{print $2}'`
DIST=`grep -w ID /etc/os-release | awk -F '=' '{print $2}'`
# if [ "$DIST" != "Ubuntu" ]; then
if [ "$DIST" != "ubuntu" -a "$DIST" != "debian" ]; then
    case "`uname`" in
        Linux*)
            MINIMUM_SUPPORTED_GLIBC_VERSION=2.17 # The GLibc supported version is defined in APP v8.0 onwards
            GLIBC_VERSION=`ldd --version | awk '/ldd/{print $NF}'`
            GLIBC_VERSION="${GLIBC_VERSION/./}"
            GLIBC_VERSION="${GLIBC_VERSION//.*/}"
            if [ $GLIBC_VERSION -lt 217 ]
            then
                echo "The GNU LIBC version is less than required \"$MINIMUM_SUPPORTED_GLIBC_VERSION\". Exit \""`basename "$0"`"\" now."
                exit 1
            fi
            ;;
        *)
            ;;
    esac
fi

# JVM symbolic link has higher priority for installation than the Environment variable JAVA_HOME
if [ -x "${BUNDLED_JAVA_HOME}" ]
then
    echo "JVM symbolic link already exists, it will be used for installation."
    # [Start] 41483: Starting v10, java 21 will be used
    # Upgrade Case
    if [ "$OS_IS_LINUX" = "1" ]; then
        case "`uname -m`" in
            x86_64 | amd64)
                if [ -d "${CBS_HOME}/${JAVA_DIR}" ]; then
                    # Delete "java" if it exists and is a directory
                    if [ -d "java" ] && [ ! -L "java" ]; then
                        echo "\"java\" is a directory. Removing it to replace with symlink..."
                        rm -rf "java"
                    fi
                    # Create or update "java" symlink to point to new java directory then delete the old java directories
                    ln -sfn "${JAVA_DIR}" "java"
                    echo "Updated symlink \"java\" to point to \"${JAVA_DIR}\"."
                    rm -rf "${CBS_HOME}/java-linux-x64" "${CBS_HOME}/java-linux-x86"
                    echo "Deleted old java directories (\"java-linux-x86\" and \"java-linux-x64\")."
                else
                    echo "Target directory \"${CBS_HOME}/${JAVA_DIR}\" does not exist. Symlink not updated."
                fi
                ;;
        esac
    fi
    # [End] 41483
else
    # Fresh Install Case
    if [ -n "${JAVA_HOME}" ]
    then
        echo "\"JAVA_HOME\" variable is set."
        BUNDLED_JAVA_HOME="${JAVA_HOME}"
    else
        if [ "$OS_IS_LINUX" = "1" ]
        then
            case "`uname -m`" in
                i[3-6]86)
                    if [ -d "${CBS_HOME}/java-linux-x86" ]
                    then
                        BUNDLED_JAVA_HOME="java-linux-x86"
                    fi
                ;;
                x86_64 | amd64)
                    # [Start] 41483: Starting v10, java 21 will be used
                    # if [ -d "${CBS_HOME}/java-linux-x64" ]; then
                    if [ -d "${CBS_HOME}/${JAVA_DIR}" ]; then
                        # BUNDLED_JAVA_HOME="java-linux-x64"
                        BUNDLED_JAVA_HOME="${JAVA_DIR}"
                    fi
                    # [End] 41483
                ;;
                *)
                    # Linux i[3-6]86, [ x86_64 | amd64 ] OS require custom JAVA_HOME
                    setup_custom_jvm
                ;;
            esac
        else
            # Non Linux OS require custom JAVA_HOME
            setup_custom_jvm
        fi
    fi
    # Create symlink for JAVA_HOME
    ln -sf "${BUNDLED_JAVA_HOME}" "java" && echo "Current Directory: \"`pwd`\"." && echo "Created symlink \"java\" to \"${BUNDLED_JAVA_HOME}\"."
fi

# Verifying if the APP's JAVA_HOME is a valid JAVA_HOME.
STRING_JAVA_VERSION="java version,openjdk version"
OUTPUT_JAVA_VERSION=`"${CBS_HOME}/java/bin/java" -version 2>&1`
OUTPUT_JVM_SUPPORT=0
BACKUP_IFS=$IFS
IFS=","
for word in $STRING_JAVA_VERSION; do
    if [ `echo "${OUTPUT_JAVA_VERSION}" | grep "${word}" | grep -cv "grep ${word}"` -le 0 ]
    then
      #echo "The Java Executable \"${CBS_HOME}/java/bin/java\" is not a valid Java Executable. Exit \""`basename "$0"`"\" now."
      continue;
    else
      OUTPUT_JVM_SUPPORT=1
      break;
    fi
done
IFS=$BACKUP_IFS
if [ $OUTPUT_JVM_SUPPORT -eq 0 ]
then
    echo "The Java Executable \"${CBS_HOME}/java/bin/java\" is not a valid Java Executable. Exit \""`basename "$0"`"\" now."
    exit 1
fi

# Verifying if the JAVA_VERSION is supported for APP installation.
# 41483: Starting v10, java 21 will be used
# MINIMUM_SUPPORTED_JVM_VERSION=1.8 # The JVM supported Version is defined in APP v6.0 onwards.
MINIMUM_SUPPORTED_JVM_VERSION="$JAVA_VERSION"
echo "Minimum supported JVM version: $MINIMUM_SUPPORTED_JVM_VERSION"
[ ! -f "$CBS_UTIL_BIN/verify-jvm-version.sh" ] && echo "The shell script \"$CBS_UTIL_BIN/verify-jvm-version.sh\" is missing." && echo "Exit \"`basename $0`\" now!" && exit 1
"$CBS_UTIL_BIN/verify-jvm-version.sh" "$CBS_HOME/java" "$MINIMUM_SUPPORTED_JVM_VERSION" 1>"/dev/null" 2>&1
if [ $? -ne 0 ]
then
    [ -L "$CBS_HOME/java" ] && rm -f "$CBS_HOME/java" && echo "Removed the Symlink \"$CBS_HOME/java\"."
    echo "The JVM version is lower than \"$MINIMUM_SUPPORTED_JVM_VERSION\" which is not supported by the CBS."
    echo "Please change the JAVA_HOME Directory and run the installation again."
    echo "Exit \"`basename $0`\" now!"
    exit 1
fi
echo "Current JVM version is supported for installation."

# -------------------------------checkLicense--------------------------
sh "${CBS_BIN}/checkLicense.sh"

# ----------------------- Install CBS As a Service -----------------------------

# Change script name to cbs
echo "Installing [ Ahsay Cloud Backup Suite ]"
CBS_SCRIPT_PATH=${CBS_BIN}
CBS_SCRIPT_NAME=cbs
CBS_SCRIPT_USER=${USER}
CBS_SYSTEMD_SRC=cbs-systemd
CBS_SYSTEMD_SVC=cbs.service
if [ "${CBS_SCRIPT_USER}" = "" ] ; then
  # handle the case if environment variable $USER is not defined (e.g. in SunOS 5.11)
  CBS_SCRIPT_USER=${LOGNAME}
fi

cd "${CBS_BIN}"

# Create the service script
case "`uname`" in
    Linux*)   CBS_SCRIPT_SRC=cbs-sysv ;;
    Solaris*) CBS_SCRIPT_SRC=cbs-sysv ;;
    SunOS*)   CBS_SCRIPT_SRC=cbs-sysv ;;
    OpenBSD*) CBS_SCRIPT_SRC=cbs-openbsd ;;
    *BSD*)    CBS_SCRIPT_SRC=cbs-bsd ;;
    **)       echo "This OS `uname` is not supported by this script! Exit `basename $0` now!"
              exit 1 ;;
esac

sed "s|@sed.script.name@|${CBS_SCRIPT_NAME}|g" < "${CBS_BIN}/${CBS_SCRIPT_SRC}" | sed "s|@sed.product.home@|${CBS_HOME}|g" | sed "s|@sed.user.name@|${CBS_SCRIPT_USER}|g" > "${CBS_BIN}/${CBS_SCRIPT_NAME}"
echo "[ Ahsay Cloud Backup Suite ] Service Script created at ${CBS_BIN}/${CBS_SCRIPT_NAME}"

sed "s|@sed.product.home@|${CBS_HOME}|g" "${CBS_BIN}/${CBS_SYSTEMD_SRC}" > "${CBS_BIN}/${CBS_SYSTEMD_SVC}"

sh "${CBS_UTIL_BIN}/install-service.sh" "${CBS_SCRIPT_PATH}/${CBS_SCRIPT_NAME}" "${CBS_SCRIPT_PATH}/${CBS_SYSTEMD_SVC}"

echo "[ Ahsay Cloud Backup Suite ] setup completed!"

# -------------------------------Migrate From previous--------------------------
echo "Migrate from previous version"
sh "${CBS_BIN}/migrate.sh"

# ----------------------------Remove hotfix from previous-----------------------
echo "Remove hotfix from previous version"
APP_WIN_ACB_HOTFIX=$CBS_HOME/download/component/app-win-acb-hotfix.7z
APP_WIN_OBM_HOTFIX=$CBS_HOME/download/component/app-win-obm-hotfix.7z
AUA_WIN_ACB_HOTFIX=$CBS_HOME/download/component/aua-win-acb-hotfix.7z
AUA_WIN_OBM_HOTFIX=$CBS_HOME/download/component/aua-win-obm-hotfix.7z
if [ -f "$APP_WIN_ACB_HOTFIX" ]; then
    echo "Remove hotfix package \"$APP_WIN_ACB_HOTFIX\""
    rm -f "$APP_WIN_ACB_HOTFIX"
fi
if [ -f "$APP_WIN_OBM_HOTFIX" ]; then
    echo "Remove hotfix package \"$APP_WIN_OBM_HOTFIX\""
    rm -f "$APP_WIN_OBM_HOTFIX"
fi
if [ -f "$AUA_WIN_ACB_HOTFIX" ]; then
    echo "Remove hotfix package \"$AUA_WIN_ACB_HOTFIX\""
    rm -f "$AUA_WIN_ACB_HOTFIX"
fi
if [ -f "$AUA_WIN_OBM_HOTFIX" ]; then
    echo "Remove hotfix package \"$AUA_WIN_OBM_HOTFIX\""
    rm -f "$AUA_WIN_OBM_HOTFIX"
fi

# ----------------------------Remove outdated files-----------------------------
echo "Remove outdated files from previous version"
rm -rf "${CBS_HOME}/build/engine-framework" "${CBS_HOME}/build/scripts" "${CBS_HOME}/download/jvm" "${CBS_HOME}/download/component"

# ------------------------------ Startup CBS -----------------------------------
echo "Startup [ Ahsay Cloud Backup Suite ]"
sh "${CBS_BIN}/startup.sh"
echo "Started [ Ahsay Cloud Backup Suite ]"

# ------------------------------ Install CBS NFS -----------------------------------
# NFS_INSTALL_SCRIPT="$CBS_HOME/nfs/bin/install.sh"
# if [ -f "$NFS_INSTALL_SCRIPT" ]; then
#     "$NFS_INSTALL_SCRIPT" "skipVerify"
# fi

# ------------------------------ Uninstall CBS NFS -----------------------------------
NFS_UNINSTALL_SCRIPT="$CBS_HOME/nfs/bin/uninstall.sh"
if [ -f "$NFS_UNINSTALL_SCRIPT" ]; then
    "$NFS_UNINSTALL_SCRIPT" "skipVerify"
fi

exit 0
